class Community:
    """Represents a Weverse Community.

    .. container:: operations

        .. describe:: x == y

            Checks if two communities are equal.

        .. describe:: x != y

            Checks if two communities are not equal.

        .. describe:: hash(x)

            Returns the community's hash.

        .. describe:: str(x)

            Returns the community's name.

    Attributes
    ----------
    data: :class:`dict`
        The raw data directly taken from the response generated by Weverse's API.
    id: :class:`int`
        The ID of the community. This can be used to fetch the dictionary
        that contains the :class:`objects.artist.Artist` objects using
        the :class:`SunverseClient.fetch_artists()` method.
    name: :class:`str`
        The official name of the community.
    alias: :class:`str`
        The alias of the community.
    code: :class:`str` | :class:`None`
        The code of the community, if any.
    url_path: :class:`str`
        The URL path to the community. E.g. lesserafim which can be used to
        form a proper URL.
    profile_image_url: :class:`str`
        The URL of the profile image of the community.
    profile_name: :class:`str`
        The name shown on the profile of the community.
    profile_cover_image_url: :class:`str`
        The URL of the profile cover image of the community.
    logo_image_url: :class:`str`
        The URL to the logo image of the community.
    home_header_image_url: :class:`str`
        The URL to the home header image of the community.
    fandom_name: :class:`str` | :class:`None`
        The name of the fandom of the community, if any.
    fan_event_url: :class:`str` | :class:`None`
        The URL used to participate in the fan events of the community, if any.
    home_gradation_colour: :class:`dict`
        The colour gradation used for the homepage for the community.
    has_membership_product: :class:`bool`
        Whether the community sells any paid membership product.
    member_count: :class:`int` | :class:`None`
        The number of users who have joined the community, if not hidden.
    birthday_artists: list[:class:`str`](?)
        The list of birthday artists of the community.
    """

    __slots__ = (
        "data",
        "id",
        "name",
        "alias",
        "code",
        "url_path",
        "profile_image_url",
        "profile_name",
        "profile_cover_image_url",
        "logo_image_url",
        "home_header_image_url",
        "fandom_name",
        "fan_event_url",
        "home_gradation_colour",
        "has_membership_product",
        "member_count",
        "birthday_artists",
    )

    def __init__(self, data: dict):
        self.data: dict = data
        self.id: int = data["communityId"]
        self.name: str = data["communityName"]
        self.alias: str = data["communityAlias"]
        self.code: str | None = data.get("artistCode")
        self.url_path: str = data["urlPath"]
        self.profile_image_url: str = data["agencyProfile"]["profileImageUrl"]
        self.profile_name: str = data["agencyProfile"]["profileName"]
        self.profile_cover_image_url: str = data["agencyProfile"][
            "profileCoverImageUrl"
        ]
        self.logo_image_url: str = data["logoImage"]
        self.home_header_image_url: str = data["homeHeaderImage"]
        self.fandom_name: str | None = data.get("fandomName")
        self.fan_event_url: str | None = data.get("fanEventUrl")
        self.home_gradation_colour: dict = data["homeGradationColor"]
        self.has_membership_product: bool = data["hasMembershipProduct"]
        self.member_count: int | None = data.get("memberCount")
        self.birthday_artists: list = data["birthdayArtists"]

    def __eq__(self, other):
        if isinstance(other, self.__class__):
            return self.id == other.id

        raise NotImplementedError

    def __repr__(self):
        return f"Community community_id={self.id}, name={self.name}"

    def __str__(self):
        return self.name

    def __hash__(self):
        return hash(self.id)
